(vl-load-com)
(setq *AREASUM_APP* "AREASUM")

;; ------------------------------------------------------------
;; Helpers
;; ------------------------------------------------------------
(defun AREASUM:_regapp ()
  (if (not (tblsearch "APPID" *AREASUM_APP*))
    (regapp *AREASUM_APP*)
  )
)

(defun AREASUM:_trim (s)
  ;; Enter => "" ; zachovej mezery, jen odstraň CR/LF
  (if s (vl-string-trim "\r\n" s) "")
)

(defun AREASUM:_getHandle (ename / h o)
  ;; 1) nejspolehlivější v progeCADu: DXF handle (group 5)
  (setq h (cdr (assoc 5 (entget ename))))
  (if (and h (/= h ""))
    h
    ;; 2) fallback přes ActiveX (když by někdy assoc 5 nebylo)
    (progn
      (setq o (vl-catch-all-apply 'vlax-ename->vla-object (list ename)))
      (if (vl-catch-all-error-p o)
        nil
        (progn
          (setq h (vl-catch-all-apply 'vla-get-Handle (list o)))
          (if (vl-catch-all-error-p h) nil h)
        )
      )
    )
  )
)
;; ------------------------------------------------------------
;; Area helpers
;; ------------------------------------------------------------
(defun AREASUM:_areaByCommand (ename / oldcmdecho oldosmode a)
  (setq oldcmdecho (getvar "CMDECHO"))
  (setq oldosmode  (getvar "OSMODE"))
  (setvar "CMDECHO" 0)
  (setvar "OSMODE" 0)

  (vl-catch-all-apply
    '(lambda ()
       (command "_.AREA" "_O" ename "")
     )
    nil
  )

  (setq a (getvar "AREA"))

  (setvar "CMDECHO" oldcmdecho)
  (setvar "OSMODE"  oldosmode)

  (if (numberp a) a 0.0)
)

(defun AREASUM:_getArea (ename / o v typ)
  (setq typ (cdr (assoc 0 (entget ename))))

  ;; Pro HATCH a REGION použij příkaz AREA (v progeCADu často nejspolehlivější)
  (if (or (= typ "HATCH") (= typ "REGION"))
    (AREASUM:_areaByCommand ename)
    (progn
      (setq o (vlax-ename->vla-object ename))

      ;; 1) Area property
      (setq v (vl-catch-all-apply 'vlax-get-property (list o 'Area)))
      (if (and (not (vl-catch-all-error-p v)) (numberp v) (> v 0.0))
        v
        ;; 2) curve area
        (progn
          (setq v (vl-catch-all-apply 'vlax-curve-getArea (list o)))
          (if (and (not (vl-catch-all-error-p v)) (numberp v) (> v 0.0))
            v
            ;; 3) poslední fallback
            (AREASUM:_areaByCommand ename)
          )
        )
      )
    )
  )
)

(defun AREASUM:_sumFromHandles (handles / h en sum)
  (setq sum 0.0)
  (foreach h handles
    (setq en (handent h))
    (if en (setq sum (+ sum (AREASUM:_getArea en))))
  )
  sum
)

;; ------------------------------------------------------------
;; XData helpers
;; ------------------------------------------------------------
(defun AREASUM:_xdataGet (edata / xd app)
  (setq xd (cdr (assoc -3 edata)))
  (if (and xd (setq app (assoc *AREASUM_APP* xd)))
    (cdr app)
    nil
  )
)

(defun AREASUM:_parseXData (xdata / prefix suffix dec handles s)
  ;; defaulty (kvůli zpětné kompatibilitě)
  (setq prefix "SOUCET PLOCH = ")
  (setq suffix "")
  (setq dec 3)
  (setq handles '())

  (foreach it xdata
    (cond
      ((and (= (car it) 1000) (wcmatch (cdr it) "PREFIX=*"))
        (setq prefix (substr (cdr it) 8))
      )
      ((and (= (car it) 1000) (wcmatch (cdr it) "SUFFIX=*"))
        (setq suffix (substr (cdr it) 8))
      )
      ((= (car it) 1070)
        (setq dec (cdr it))
      )
      ((and (= (car it) 1000) (wcmatch (cdr it) "H:*"))
        (setq s (substr (cdr it) 3))
        (if (> (strlen s) 0) (setq handles (cons s handles)))
      )
    )
  )
  ;; vrací: prefix dec handles suffix
  (list prefix dec (reverse handles) suffix)
)

(defun AREASUM:_setXDataOnEntity (ename prefix dec handles suffix / ed xdata)
  ;; nastaví XData přes entmod (bezpečně pro progeCAD)
  (AREASUM:_regapp)
  (setq ed (entget ename '("*")))

  (if (null suffix) (setq suffix ""))
  (if (null prefix) (setq prefix ""))

  (setq xdata
    (cons -3
      (list
        (append
          (list
            *AREASUM_APP*
            (cons 1000 (strcat "PREFIX=" prefix))
            (cons 1000 (strcat "SUFFIX=" suffix))
            (cons 1070 dec)
          )
          (mapcar '(lambda (h) (cons 1000 (strcat "H:" h))) handles)
        )
      )
    )
  )

  ;; odeber případné staré -3 a přidej nové
  (setq ed (vl-remove (assoc -3 ed) ed))
  (entmod (append ed (list xdata)))
  (entupd ename)
  T
)

(defun AREASUM:_replacePSOnlyInXData (xdata newPrefix newSuffix / out it hasP hasS)
  (setq out '())
  (setq hasP nil)
  (setq hasS nil)

  (foreach it xdata
    (cond
      ((and (= (car it) 1000) (wcmatch (cdr it) "PREFIX=*"))
        (setq hasP T)
        (setq out (append out (list (cons 1000 (strcat "PREFIX=" newPrefix)))))
      )
      ((and (= (car it) 1000) (wcmatch (cdr it) "SUFFIX=*"))
        (setq hasS T)
        (setq out (append out (list (cons 1000 (strcat "SUFFIX=" newSuffix)))))
      )
      (t
        (setq out (append out (list it)))
      )
    )
  )

  ;; doplň, pokud chybělo
  (if (not hasP)
    (setq out (append (list (cons 1000 (strcat "PREFIX=" newPrefix))) out))
  )
  (if (not hasS)
    (setq out (append out (list (cons 1000 (strcat "SUFFIX=" newSuffix)))))
  )

  out
)

;; ------------------------------------------------------------
;; Update helpers
;; ------------------------------------------------------------
(defun AREASUM:_isNumChar (ch)
  (or (and (>= ch 48) (<= ch 57))  ; 0-9
      (= ch 46)                    ; .
      (= ch 44)                    ; ,
      (= ch 45)                    ; -
      (= ch 43)                    ; +
  )
)

(defun AREASUM:_isAlphaChar (ch)
  ;; ASCII písmena A-Z a-z
  (or (and (>= ch 65) (<= ch 90))
      (and (>= ch 97) (<= ch 122)))
)


(defun AREASUM:_splitByLastNumber (s / n i inNum end start ch prefix suffix preCh postCh ok)
  ;; Vrátí (list prefix suffix) podle posledního "čísla" v textu,
  ;; ale ignoruje čísla nalepená na písmena (např. mm2).
  (if (or (null s) (= s "")) nil
    (progn
      (setq n (strlen s))
      (setq i n)

      (while (> i 0)
        ;; 1) najdi konec číselného bloku (odzadu)
        (setq inNum nil)
        (setq end nil)
        (setq start nil)

        (while (> i 0)
          (setq ch (ascii (substr s i 1)))
          (if (AREASUM:_isNumChar ch)
            (progn
              (if (not inNum)
                (progn (setq inNum T) (setq end i))
              )
            )
            (progn
              (if inNum
                (progn (setq start (1+ i)) (setq i 0))
              )
            )
          )
          (if (> i 0) (setq i (1- i)))
        )

        ;; pokud text končí číslem
        (if (and inNum (null start)) (setq start 1))

        (if (and start end (<= start end))
          (progn
            ;; 2) ověř hranice: nesmí být nalepené na písmena
            (setq preCh  nil)
            (setq postCh nil)

            (if (> start 1)
              (setq preCh (ascii (substr s (1- start) 1)))
            )
            (if (< end n)
              (setq postCh (ascii (substr s (1+ end) 1)))
            )

            (setq ok T)
            (if (and preCh (AREASUM:_isAlphaChar preCh)) (setq ok nil))
            (if (and postCh (AREASUM:_isAlphaChar postCh)) (setq ok nil))

            (if ok
              (progn
                (setq prefix (substr s 1 (1- start)))
                (setq suffix (substr s (1+ end) (- n end)))
                (setq i 0) ; ukonči hledání úspěchem
                (setq s (list prefix suffix))
              )
              ;; jinak pokračuj hledáním dřívějšího čísla:
              (setq i (1- start))
            )
          )
          (setq i 0) ;; žádné číslo nenalezeno
        )
      )

      (if (and (listp s) (= (length s) 2)) s nil)
    )
  )
)

;; ------------------------------------------------------------
;; Update TEXT entity
;; ------------------------------------------------------------
(defun AREASUM:_updateTextEntity (ename / ed xd parsed prefix dec handles suffix sum newnum cur split newP newS
                                 newstr newXData x3)
  (setq ed (entget ename '("*")))
  (setq xd (AREASUM:_xdataGet ed))
  (if (not xd)
    nil
    (progn
      (setq parsed  (AREASUM:_parseXData xd))
      (setq prefix  (nth 0 parsed))
      (setq dec     (nth 1 parsed))
      (setq handles (nth 2 parsed))
      (setq suffix  (nth 3 parsed))

      ;; spočti nový součet a číslo
      (setq sum    (AREASUM:_sumFromHandles handles))
      (setq newnum (rtos sum 2 dec))

      ;; zkus vzít prefix/suffix z aktuálního textu (poslední číslo)
      (setq cur (cdr (assoc 1 ed)))
      (setq split (AREASUM:_splitByLastNumber cur))

      (if split
        (progn
          (setq newP (car split))
          (setq newS (cadr split))

          ;; pokud se P/S liší od XData, zapiš je do XData
          (if (or (/= newP prefix) (/= newS suffix))
            (progn
              (setq newXData (AREASUM:_replacePSOnlyInXData xd newP newS))
              (setq x3 (cons -3 (list (append (list *AREASUM_APP*) newXData))))
              (setq ed (vl-remove (assoc -3 ed) ed))
              (entmod (append ed (list x3)))
              (entupd ename)

              ;; ať pokračujeme s novými hodnotami
              (setq prefix newP)
              (setq suffix newS)
            )
          )
        )
      )

      ;; slož nový text
      (setq newstr (strcat prefix newnum suffix))

      ;; přepiš group 1
      (setq ed (entget ename '("*")))
      (if (assoc 1 ed)
        (setq ed (subst (cons 1 newstr) (assoc 1 ed) ed))
        (setq ed (append ed (list (cons 1 newstr))))
      )

      (entmod ed)
      (entupd ename)
      T
    )
  )
)


;; ------------------------------------------------------------
;; Commands
;; ------------------------------------------------------------
(defun c:AREASUMLINK (/ ss i en sum dec ht rot ins prefix suffix txt handles ent)
  (AREASUM:_regapp)

  ;; Prefix/Suffix: Enter = prázdné (nemusíš dávat mezerník)
  (setq prefix (AREASUM:_trim (getstring T "\nPrefix textu <>: ")))
  (setq suffix (AREASUM:_trim (getstring T "\nSuffix textu (např. \" mm2\") <>: ")))

  (prompt "\nVyber objekty, ze kterých chceš sečíst plochu (uzavřené křivky/šrafy/regiony apod.): ")
  (setq ss (ssget))
  (if (not ss)
    (progn (prompt "\nNic nevybráno.") (princ))
    (progn
      (setq sum 0.0)
      (setq handles '())
      (setq i 0)
      (while (< i (sslength ss))
        (setq en (ssname ss i))
        (setq handles (append handles (list (AREASUM:_getHandle en))))
        (setq sum (+ sum (AREASUM:_getArea en)))
        (setq i (1+ i))
      )

      (prompt (strcat "\nSoučet ploch = " (rtos sum 2 3)))

      (setq dec (getint "\nPočet desetinných míst <3>: "))
      (if (null dec) (setq dec 3))

      (setq ht (getdist "\nVýška textu <2.5>: "))
      (if (null ht) (setq ht 2.5))

      (setq rot (getangle "\nRotace textu <0>: "))
      (if (null rot) (setq rot 0.0))

      (setq ins (getpoint "\nZadej bod vložení textu: "))
      (if (not ins)
        (prompt "\nVložení zrušeno.")
        (progn
          (setq txt (strcat prefix (rtos sum 2 dec) suffix))

          ;; 1) vytvoř TEXT bez XData
          (setq ent
            (entmakex
              (list
                (cons 0 "TEXT")
                (cons 8 (getvar "CLAYER"))
                (cons 10 ins)
                (cons 40 ht)
                (cons 1 txt)
                (cons 50 rot)
                (cons 7 (getvar "TEXTSTYLE"))
              )
            )
          )

          (if (not ent)
            (prompt "\nNepodařilo se vytvořit text.")
            (progn
              ;; 2) doplň XData až dodatečně (včetně suffixu)
              (AREASUM:_setXDataOnEntity ent prefix dec handles suffix)
              (prompt "\nLinked text vytvořen. Aktualizace: AREASUMUPD.")
            )
          )
        )
      )
    )
  )
  (princ)
)

(defun c:AREASUMUPD (/ mode ss i en cnt)
  (initget "V A")
  (setq mode (getkword "\nAktualizace: [V]yber texty / [A]ll (vse) <A>: "))
  (if (null mode) (setq mode "A"))

  (setq cnt 0)

  (cond
    ((= (strcase mode) "V")
      (prompt "\nVyber jeden nebo více TEXTů, které jsou AREASUM-linked: ")
      (setq ss (ssget '((0 . "TEXT"))))
    )
    (T
      (setq ss (ssget "X" '((0 . "TEXT"))))
    )
  )

  (if ss
    (progn
      (setq i 0)
      (while (< i (sslength ss))
        (setq en (ssname ss i))
        (if (AREASUM:_updateTextEntity en)
          (setq cnt (1+ cnt))
        )
        (setq i (1+ i))
      )
    )
  )

  (prompt (strcat "\nAktualizováno textů: " (itoa cnt)))
  (princ)
)

(defun c:AREASUMINFO (/ e ed xd parsed prefix dec handles n bad ssAll h en act)
  (setq e (car (entsel "\nVyber AREASUM text/objekt: ")))
  (if (not e)
    (progn (prompt "\nNic nevybráno.") (princ))
    (progn
      (setq ed (entget e '("*")))
      (setq xd (AREASUM:_xdataGet ed))
      (if (not xd)
        (prompt "\nVybraný objekt nemá AREASUM XData.")
        (progn
          (setq parsed  (AREASUM:_parseXData xd))
          (setq prefix  (nth 0 parsed))
          (setq dec     (nth 1 parsed))
          (setq handles (nth 2 parsed))
          (setq suffix  (nth 3 parsed))

          (setq n (length handles))
          (prompt (strcat
                    "\n--- AREASUM INFO ---"
                    "\nPrefix: " prefix
					"\nSuffix: " suffix
                    "\nDesetinná místa: " (itoa dec)
                    "\nPočet objektů: " (itoa n)))

          (setq bad 0)
          (foreach h handles
            (if (not (handent h)) (setq bad (1+ bad)))
          )
          (if (> bad 0)
            (prompt (strcat "\nPozor: neexistujících objektů: " (itoa bad)))
          )

          (setq ssAll (ssadd))
          (ssadd e ssAll)
          (foreach h handles
            (setq en (handent h))
            (if en (ssadd en ssAll))
          )

          (initget "O Z P N")
          (setq act (getkword "\nAkce: [O]bojí / [Z]výraznit / [P]řejít (zoom) / [N]ic <O>: "))
          (if (null act) (setq act "O"))

          (cond
            ((= act "N") (princ))
            ((= act "Z")
              (sssetfirst nil ssAll)
              (prompt "\nZvýrazněno: text + navázané objekty.")
            )
            ((= act "P")
              (if (> (sslength ssAll) 0)
                (progn
                  (command "_.ZOOM" "_Object" ssAll "")
                  (sssetfirst nil ssAll)
                  (prompt "\nZoom na text + navázané objekty, výběr ponechán.")
                )
                (prompt "\nNení na co zoomovat (výběr je prázdný).")
              )
            )
            ((= act "O")
              (sssetfirst nil ssAll)
              (if (> (sslength ssAll) 0)
                (progn
                  (command "_.ZOOM" "_Object" ssAll "")
                  (sssetfirst nil ssAll)
                  (prompt "\nZvýrazněno + zoom na text + navázané objekty (výběr ponechán).")
                )
                (prompt "\nZvýrazněno, ale není na co zoomovat (výběr je prázdný).")
              )
            )
          )
        )
      )
    )
  )
  (princ)
)

(defun c:AREASUMBACK (/ obj ed typ oh ss i t edT xd parsed handles ssTxt ssAll act
                         prefix dec suffix n bad h en)

  (setq obj (car (entsel "\nVyber objekt (nebo AREASUM text) pro nalezení navázaných textů: ")))
  (if (not obj)
    (progn (prompt "\nNic nevybráno.") (princ))
    (progn
      (setq ed (entget obj '("*")))
      (setq typ (cdr (assoc 0 ed)))

      ;; ------------------------------------------------------------
      ;; Když uživatel omylem vybere TEXT:
      ;; - pokud má AREASUM XData -> chovej se jako INFO
      ;; - pokud nemá -> upozorni a skonči
      ;; ------------------------------------------------------------
      (if (= typ "TEXT")
        (progn
          (setq xd (AREASUM:_xdataGet ed))
          (if (not xd)
            (progn
              (prompt "\nVybral jsi TEXT, ale není AREASUM-linked (nemá XData). Vyber prosím zdrojový objekt (křivku/šrafu/region).")
              (princ)
            )
            (progn
              ;; INFO režim (zjednodušeně jako AREASUMINFO)
              (setq parsed  (AREASUM:_parseXData xd))
              (setq prefix  (nth 0 parsed))
              (setq dec     (nth 1 parsed))
              (setq handles (nth 2 parsed))
              (setq suffix  (nth 3 parsed))

              (setq n (length handles))
              (prompt (strcat
                        "\n--- AREASUM INFO (vybral jsi text) ---"
                        "\nPrefix: " prefix
                        "\nSuffix: " suffix
                        "\nDesetinná místa: " (itoa dec)
                        "\nPočet objektů: " (itoa n)))

              (setq bad 0)
              (foreach h handles
                (if (not (handent h)) (setq bad (1+ bad)))
              )
              (if (> bad 0)
                (prompt (strcat "\nPozor: neexistujících objektů: " (itoa bad)))
              )

              ;; připrav selection set: text + navázané objekty
              (setq ssAll (ssadd))
              (ssadd obj ssAll)
              (foreach h handles
                (setq en (handent h))
                (if en (ssadd en ssAll))
              )

              (initget "O Z P N")
              (setq act (getkword "\nAkce: [O]bojí / [Z]výraznit / [P]řejít (zoom) / [N]ic <O>: "))
              (if (null act) (setq act "O"))

              (cond
                ((= act "N") (princ))
                ((= act "Z")
                  (sssetfirst nil ssAll)
                  (prompt "\nZvýrazněno: text + navázané objekty.")
                )
                ((= act "P")
                  (if (> (sslength ssAll) 0)
                    (progn
                      (command "_.ZOOM" "_Object" ssAll "")
                      (sssetfirst nil ssAll)
                      (prompt "\nZoom na text + navázané objekty, výběr ponechán.")
                    )
                    (prompt "\nNení na co zoomovat (výběr je prázdný).")
                  )
                )
                ((= act "O")
                  (sssetfirst nil ssAll)
                  (if (> (sslength ssAll) 0)
                    (progn
                      (command "_.ZOOM" "_Object" ssAll "")
                      (sssetfirst nil ssAll)
                      (prompt "\nZvýrazněno + zoom na text + navázané objekty (výběr ponechán).")
                    )
                    (prompt "\nZvýrazněno, ale není na co zoomovat (výběr je prázdný).")
                  )
                )
              )
              (princ)
            )
          )
        )
        ;; ------------------------------------------------------------
        ;; Jinak: vybraný je "normální" objekt -> najdi navázané texty (BACK)
        ;; ------------------------------------------------------------
        (progn
          ;; Handle objektu (pro kompatibilitu ponechám assoc 5; pokud máš helper AREASUM:_getHandle, klidně sem dej ten)
          (setq oh (AREASUM:_getHandle obj))

          ;; ProgeCAD: filtrace přes (1001 . APP) nemusí fungovat -> vem všechny TEXTy a filtruj až přes entget
          (setq ss (ssget "X" '((0 . "TEXT"))))

          (setq ssTxt (ssadd))
          (if ss
            (progn
              (setq i 0)
              (while (< i (sslength ss))
                (setq t (ssname ss i))
                (setq edT (entget t '("*")))
                (setq xd (AREASUM:_xdataGet edT))
                (if xd
                  (progn
                    (setq parsed  (AREASUM:_parseXData xd))
                    (setq handles (nth 2 parsed))
                    (if (member oh handles)
                      (ssadd t ssTxt)
                    )
                  )
                )
                (setq i (1+ i))
              )
            )
          )

          (if (= (sslength ssTxt) 0)
            (prompt "\nNenalezen žádný AREASUM text navázaný na tento objekt.")
            (progn
              (prompt (strcat "\nNalezeno textů: " (itoa (sslength ssTxt))))

              (setq ssAll (ssadd))
              (ssadd obj ssAll)
              (setq i 0)
              (while (< i (sslength ssTxt))
                (ssadd (ssname ssTxt i) ssAll)
                (setq i (1+ i))
              )

              (initget "O Z P N")
              (setq act (getkword "\nAkce: [O]bojí / [Z]výraznit / [P]řejít (zoom) / [N]ic <O>: "))
              (if (null act) (setq act "O"))

              (cond
                ((= act "N") (princ))
                ((= act "Z")
                  (sssetfirst nil ssAll)
                  (prompt "\nZvýrazněno: objekt + navázané texty.")
                )
                ((= act "P")
                  (command "_.ZOOM" "_Object" ssAll "")
                  (sssetfirst nil ssAll)
                  (prompt "\nZoom na objekt + navázané texty, výběr ponechán.")
                )
                ((= act "O")
                  (sssetfirst nil ssAll)
                  (command "_.ZOOM" "_Object" ssAll "")
                  (sssetfirst nil ssAll)
                  (prompt "\nZvýrazněno + zoom na objekt + navázané texty (výběr ponechán).")
                )
              )
            )
          )
        )
      )
    )
  )
  (princ)
)


;; ------------------------------------------------------------
;; Relink support (mění jen H:..., prefix/suffix/dec zůstává)
;; ------------------------------------------------------------
(defun AREASUM:_replaceHandlesInXData (xdata newHandles / out)
  (setq out '())
  (foreach it xdata
    (cond
      ((and (= (car it) 1000) (wcmatch (cdr it) "H:*")) nil)  ;; vyhoď H:
      ((and (= (car it) 1000) (wcmatch (cdr it) "T:*")) nil)  ;; vyhoď T: (kdyby bylo)
      (T (setq out (append out (list it))))
    )
  )
  (setq out (append out (mapcar '(lambda (h) (cons 1000 (strcat "H:" h))) newHandles)))
  out
)

(defun c:AREASUMRELINK (/ e ed xd ss i obj newHandles newXData x3)
  (setq e (car (entsel "\nVyber existující AREASUM-linked TEXT k přenavázání: ")))
  (if (not e)
    (progn (prompt "\nNic nevybráno.") (princ))
    (progn
      (setq ed (entget e '("*")))
      (setq xd (AREASUM:_xdataGet ed))

      (if (not xd)
        (progn
          (prompt "\nVybraný objekt NENÍ AREASUM-linked (nemá XData).")
          (princ)
        )
        (progn
          (prompt "\nVyber NOVÉ objekty, na které se má součet vázat: ")
          (setq ss (ssget))
          (if (not ss)
            (prompt "\nNevybráno, zrušeno.")
            (progn
              (setq newHandles '())
              (setq i 0)
              (while (< i (sslength ss))
                (setq obj (ssname ss i))
                (setq newHandles (append newHandles (list (AREASUM:_getHandle obj))))
                (setq i (1+ i))
              )

              (setq newXData (AREASUM:_replaceHandlesInXData xd newHandles))
              (setq x3 (cons -3 (list (append (list *AREASUM_APP*) newXData))))

              (setq ed (vl-remove (assoc -3 ed) ed))
              (entmod (append ed (list x3)))
              (entupd e)

              (AREASUM:_updateTextEntity e)
              (prompt "\nHotovo: vazba přenastavena a text aktualizován (formát zachován).")
            )
          )
        )
      )
    )
  )
)
(princ)